// BlobTransformable.h
// Definit la classe des blobs transformables (par une matrice homogene)

#ifndef V3D_BLOBS_BLOB_TRANSFORMABLE_H_INCLUDED
#define V3D_BLOBS_BLOB_TRANSFORMABLE_H_INCLUDED

#include "BlobBase.h"

namespace V3D {


/*
 Classe definissant les blobs transformables par une transformation affine.

 Des optimisations et utilitaires sont mises en place pour que la creation des classes derivees
 soient la plus simple possible.

 Pour fonctionner, une classe derivee doit definir GetBaseShapeSqrDist et ConvertBaseShapePosToGradient.
 Le resultat est fonctionnel, mais pas tres optimise

  - EarlyRejectPoint et/ou EarlyRejectBaseShapePoint servent a optimiser

  - EnergyBaseShapeAddAtRectangle peut etre redefinie pour reduire la grille a un sous-ensemble 
        pour lequel les potentiels sont non nuls


Certaines methodes utilisent beaucoup des methodes simples (GetBaseShapeSqrDist, etc...).
Pour eviter les chutes de performances (tres nombreux appels de methodes virtuelles),
des methodes template specialisees par classe de blob sont creees, et utilisees dans les classes derivees.
La macro BLOBTRANSFORMABLE_STD_OVERRIDES automatise tout cela.

*/
	
/////////////////////////////////////////////////////////////////////////////////////////
class BlobTransformable : public BlobBase
{
	typedef BlobBase parent;
public:
	// Construction / Destruction
	BlobTransformable( float fCenterEnergy, 
	                   const Vector3D& vcPos,
	                   float fInfluenceRange,
	                   EEvalType etStyle = BlobBase::etPolynom4 );
	
	virtual ~BlobTransformable() {}

		// Accesseurs

	// Recuperation de la matrice transformant la forme de base en la forme deformee
	inline const MatrixTransf& GetTransform() const;

	// Mise en place de la matrice de transformation
	void SetTransform( const MatrixTransf& transf);

	// Changement uniquement de la partie position de la matrice.
	void SetPosition( const Vector3D& pos);

	// Changement uniquement de la partie scale de la matrice pour obtenir l'etendue fournie en parametre.
	void SetInfluenceRange(float fInfl);



	// Methodes issues de BlobBase a redefinir par les filles ;
	//	virtual BlobBase* Clone() const = 0;

	// Calcule a partir de la boite definie par vcBoxMin et vcBoxMax, son sous-ensemble minimum contenant le blob
	// Retourne true si cette boite minimum existe (volume non nul)
	//	virtual bool ComputeMinEnclosingBoxAligned(Vector3D& vcMin, Vector3D& vcMax, const Vector3D& vcBoxMin, const Vector3D& vcBoxMax) const;

	// Test de rejet d'un point dans le repere global
	virtual bool  EarlyRejectPoint( const Vector3D& /* pos */) const          {return false;}

	// Test de rejet d'un point dans le repere local
	virtual bool  EarlyRejectBaseShapePoint( const Vector3D& /* pos */) const {return false;}

	// Calcul de la distance d'un point (exprime dans le repere local) a la forme de base (avant transformation) du blob
	virtual float GetBaseShapeSqrDist( const Vector3D& pos) const      = 0;
	
	// Conversion d'un vecteur position dans le repere local vers le gradient associe
	virtual void  ConvertBaseShapePosToGradient( Vector3D& vcOffs) const = 0;

	

	// Ajout de l'energie emise par le blob aux sommets d'un treillis dans un tableau (3 dimensions) de valeurs
	// Ne pas le redefinir dans les classes derivees : redefinir EnergyBaseShapeAddAtRectangle a la place
	virtual void EnergyAddAtGrid( float* pafAddEnergies, 
	                             const Vector3D& vcStart, const Vector3D& vcSteps,
	                             int32 nStepsX, int32 nStepsY, int32 nStepsZ) const;


	// Methodes redefinies automatiquement par la macro BLOBTRANSFORMABLE_STD_OVERRIDES

	// Recherche des indices de vertex inclus dans le champ d'influence du blob
//	virtual void FindInfluencedVerticesSubArray( VertexIndexArray& anVtxInfluencedIdx, 
//	                                            const Vect3DArray& aPos,
//	                                            int32 nStart, int32 nCount) const = 0;

	// Calcul des gradients d'un sous-tableau de vertex 
	// et determination a la volee des indices des vertex inclus dans le champ d'influence du blob
//	virtual void GradientAddAtSubArrayFillInfluenced( Vect3fArray& aGrads,
//	                                                  const Vect3DArray& aPos,
//	                                                  VertexIndexArray& anInflVtxToFind,
//	                                                  int32 nStart, int32 nCount) const = 0;

	// Calcul des gradients d'un tableau de vertex (uniquement la liste des vertices influences par le blob)
//	virtual void GradientAddAtInfluencedVertices( Vect3fArray& aGrads, 
//	                                              const Vect3DArray& aPos,
//	                                              const VertexIndexArray& anInflVtx) const = 0;



	// Redefinie a partir des nouvelles methodes (ne pas trop utiliser, car appels virtuels)
//	virtual inline void GradientAddAtPos( Vector3f& vcGrad, const Vector3D& pos) const = 0;



	// Fonction redefinie pour tenir compte de la matrice de transformation.
	// Pas vraiment optimale (appels virtuels).
	virtual inline bool EnergyAddAtPos( float& fDest, const Vector3D& pos) const;

protected:
	// Accesseur de la transformation inverse du blob (repere global -> repere local)
	inline const MatrixTransf& GetInverseTransform() const;

	// Calcul de l'energie emise par le blob a une position exprimee dans le repere local
	inline bool EnergyAddAtPosBaseShape( float& fDest, const Vector3D& pos) const;

	
	
	// Ajout de l'energie emise par le blob aux sommets d'un treillis 2D dans un tableau (2 dimensions) de valeurs
	// Peut etre redefinie pour tenir compte d'infos supplementaires sur le blob afin de ne pas visiter tous les sommets.
    
	virtual void EnergyBaseShapeAddAtRectangle ( float* pafAddEnergies,    // Tableau des valeurs energetiques de la grille
										 const Vector3D& vcRectStart,  // Origine du rectangle
										 const Vector3D& vcRectAxisX,  // Axe X du rectangle (sert d'unite de separation entre deux calculs d'energie)
										 const Vector3D& vcRectAxisY,  // Axe Y du rectangle (sert d'unite de separation entre deux calculs d'energie)
                                         int32 nLinesCount,                    // Nombre de subdivisions en Y dans le rectangle
										 int32 nLineLength,                    // Nombre de subdivisions en X dans le rectangle
										 int32 nSizeX                          // Nombre de subdivisions en X dans la grille
												) const;

	
	// Fonction virtuelle definie par BlobBase, a redefinir par les filles
	//virtual bool ComputeBlobBounds(Vector3D& vcBlobMin, Vector3D& vcBlobMax) const;

	// A redefinir par les filles si une operation est a effectuer 
	// lorsqu'une transformation a eu lieu (ex : precalculs, etc...)
	virtual void OnTransformChanged() {}


	// Methodes template utilitaires permettant aux classes filles de definir les methodes virtuelles
	// - GradientAddAtPos
	// - GradientAddAtInfluencedVertices,
	// - GradientAddAtSubArrayFillInfluenced et 
	// - FindInfluencedVerticesSubArray 
	// - EnergyGetAtInfluencedVertices
	// utilisees dans la macro BLOBTRANSFORMABLE_STD_OVERRIDES
	// afin que les classes derivees redefinissent toutes ces methodes en une seule ligne
	
	// Ces methodes sont specialisees pour eviter des appels virtuels couteux trop nombreux
	// (les gains en performances sont assez importants)

	// Redefinie a partir des nouvelles methodes (ne pas trop utiliser, car appels virtuels)
	template <class BLOBCLASS>
		inline bool GradientAddAtPosClass( Vector3f& vcGrad, const Vector3D& pos) const;

	template <class BLOBCLASS>
		void GradientAddAtInfluencedVerticesClass( Vect3fArray& aGrads, const Vect3DArray& aPos,
		                                           const VertexIndexArray& anInflVtx) const;

	template <class BLOBCLASS>
		void GradientAddAtSubArrayFillInfluencedClass( Vect3fArray& aGrads, const Vect3DArray& aPos,
		                                               VertexIndexArray& anInflVtxToFind,
		                                               int32 nStart, int32 nCount) const;

	template <class BLOBCLASS>
		void FindInfluencedVerticesSubArrayClass( VertexIndexArray& anVtxInfluencedIdx,
		                                          const Vect3DArray& aPos,
		                                          int32 nStart, int32 nCount) const;

	template <class BLOBCLASS>
		inline void EnergyGetAtInfluencedVerticesClass( VertexInfluenceArray& afVals, const Vect3DArray& aPos,
		                                                const VertexIndexArray& anVertexIdx) const;

private:

	template <class BLOBCLASS, int EVALTYPE>
		inline bool GradientAddAtPosConfig( Vector3f& vcGrad, const Vector3D& pos) const;


	template <class BLOBCLASS, int EVALTYPE>
		void GradientAddAtInfluencedVerticesConfig( Vect3fArray& aGrads, const Vect3DArray& aPos,
	                                                const VertexIndexArray& anInflVtx) const;

	template <class BLOBCLASS, int EVALTYPE>
		void GradientAddAtSubArrayFillInfluencedConfig( Vect3fArray& aGrads, const Vect3DArray& aPos,
	                                                    VertexIndexArray& anInflVtxToFind,
	                                                    int32 nStart, int32 nCount) const;

	template <class BLOBCLASS, int EVALTYPE>
		inline void EnergyGetAtInfluencedVerticesConfig( VertexInfluenceArray& afVals,const Vect3DArray& aPos,
		                                                 const VertexIndexArray& anVertexIdx) const;



private:
	MatrixTransf m_matTransfBaseShape;	        // Matrice de transformation de la forme de base -> forme transformee
	MatrixTransf m_matTransfBaseShapeInv;	    // Matrice de transformation de la forme transformee -> forme de base
	MatrixTransf m_matTransfBaseShapeNormals;   // Matrice de transformation des normales de forme de base -> forme transformee
};




// Macro a utiliser dans les classes derivees



#define BLOBTRANSFORMABLE_STD_OVERRIDES( BLOBCLASS )                                                     \
                                                                                                         \
  BLOBBASE_STD_OVERRIDES( BLOBCLASS )                                                                    \
                                                                                                         \
  virtual void FindInfluencedVerticesSubArray( VertexIndexArray& anVtxInfluencedIdx,                     \
                                                const Vect3DArray& aPos,                         \
                                                int32 nStart, int32 nCount) const                        \
  {                                                                                                      \
    FindInfluencedVerticesSubArrayClass<BLOBCLASS> (anVtxInfluencedIdx, aPos, nStart, nCount);           \
  }                                                                                                      \
                                                                                                         \
  virtual void GradientAddAtSubArrayFillInfluenced( Vect3fArray& aGrads, const Vect3DArray& aPos,  \
	                                                     VertexIndexArray& anInflVtxToFind,              \
	                                                     int32 nStart, int32 nCount) const               \
  {                                                                                                      \
	GradientAddAtSubArrayFillInfluencedClass<BLOBCLASS> (aGrads, aPos, anInflVtxToFind, nStart, nCount); \
  }                                                                                                      \
                                                                                                         \
  virtual void GradientAddAtInfluencedVertices( Vect3fArray& aGrads,                             \
                                                 const Vect3DArray& aPos,                        \
                                                 const VertexIndexArray& anInflVtx) const                \
  {                                                                                                      \
	GradientAddAtInfluencedVerticesClass<BLOBCLASS> (aGrads, aPos, anInflVtx);                           \
  }                                                                                                      \
                                                                                                         \
  virtual inline bool GradientAddAtPos( Vector3f& vcGrad, const Vector3D& pos) const     \
  {                                                                                                      \
	return GradientAddAtPosClass<BLOBCLASS> (vcGrad, pos);                                               \
  }                                                                                                      \
                                                                                                         \
  virtual void EnergyGetAtInfluencedVertices( VertexInfluenceArray& afVals,                              \
                                              const Vect3DArray& aPos,                           \
                                              const VertexIndexArray& anVertexIdx) const                 \
  {                                                                                                      \
	EnergyGetAtInfluencedVerticesClass<BLOBCLASS> (afVals, aPos, anVertexIdx );                          \
  }                                                                                                      

  

//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////


inline const MatrixTransf& BlobTransformable::GetTransform() const
{
	return m_matTransfBaseShape;
}

                           /////////////////////////////

inline const MatrixTransf& BlobTransformable::GetInverseTransform() const
{
	return m_matTransfBaseShapeInv;
}

                           /////////////////////////////


inline bool BlobTransformable::EnergyAddAtPos( float& fDest, const Vector3D& pos) const
{
	Vector3D vcOffs = pos;
	m_matTransfBaseShapeInv.Transform(vcOffs); 
/*
	// inline manuel (le compilateur ne garantit pas le inline) de   vcOffs = m_matTransfBaseShapeInv * pos
	vcOffs.x = m_matTransfBaseShapeInv[MatrixTransf::M11] * pos.x +
	           m_matTransfBaseShapeInv[MatrixTransf::M12] * pos.y +
	           m_matTransfBaseShapeInv[MatrixTransf::M13] * pos.z +
	           m_matTransfBaseShapeInv[MatrixTransf::M14];
	vcOffs.y = m_matTransfBaseShapeInv[MatrixTransf::M21] * pos.x +
	           m_matTransfBaseShapeInv[MatrixTransf::M22] * pos.y +
	           m_matTransfBaseShapeInv[MatrixTransf::M23] * pos.z +
	           m_matTransfBaseShapeInv[MatrixTransf::M24];
	vcOffs.z = m_matTransfBaseShapeInv[MatrixTransf::M31] * pos.x +
	           m_matTransfBaseShapeInv[MatrixTransf::M32] * pos.y +
	           m_matTransfBaseShapeInv[MatrixTransf::M33] * pos.z +
	           m_matTransfBaseShapeInv[MatrixTransf::M34];
*/
	return EnergyAddAtPosBaseShape(fDest, vcOffs );
}

                           /////////////////////////////


// Retourne l'energie generee par l'ellipsoide au point 'pos'
inline bool BlobTransformable::EnergyAddAtPosBaseShape( float& fDest, const Vector3D& pos) const
{
	if( ! EarlyRejectBaseShapePoint(pos) )
	{
		float fRadDivInflSqr = GetBaseShapeSqrDist(pos);

		if( fRadDivInflSqr < 1.f )
		{
			fDest += EnergyNotZeroAtSqrDist(fRadDivInflSqr);
			return true;
		}
	}
	return false;
}

                           /////////////////////////////


template <class BLOBCLASS>
inline void BlobTransformable::EnergyGetAtInfluencedVerticesClass( VertexInfluenceArray& afVals,
                                                                   const Vect3DArray& aPos,
                                                                   const VertexIndexArray& anVertexIdx) const
{
	switch( GetStyle() )
	{
	case etQuadratic : EnergyGetAtInfluencedVerticesConfig<BLOBCLASS, etQuadratic> (afVals, aPos, anVertexIdx); break;
	case etPolynom4  : EnergyGetAtInfluencedVerticesConfig<BLOBCLASS, etPolynom4 > (afVals, aPos, anVertexIdx); break;
	case etPolynom6  : EnergyGetAtInfluencedVerticesConfig<BLOBCLASS, etPolynom6 > (afVals, aPos, anVertexIdx); break;
	default: 
		assert(0); 
		break;
	}
}




template <class BLOBCLASS>
inline bool BlobTransformable::GradientAddAtPosClass( Vector3f& vcGrad, const Vector3D& pos) const
{
	bool bResult;
	switch( GetStyle() )
	{
	case etQuadratic : bResult = GradientAddAtPosConfig<BLOBCLASS, etQuadratic> ( vcGrad, pos ); break;
	case etPolynom4  : bResult = GradientAddAtPosConfig<BLOBCLASS, etPolynom4 > ( vcGrad, pos ); break;
	case etPolynom6  : bResult = GradientAddAtPosConfig<BLOBCLASS, etPolynom6 > ( vcGrad, pos ); break;
	default: 
		bResult = false;
		assert(0); 
		break;
	}
	return bResult;
}


                           /////////////////////////////


template <class BLOBCLASS>
void BlobTransformable::GradientAddAtInfluencedVerticesClass( Vect3fArray& aGrads, const Vect3DArray& aPos,
	                                                            const VertexIndexArray& anInflVtx) const
{
	switch( GetStyle() )
	{
	case etQuadratic : GradientAddAtInfluencedVerticesConfig<BLOBCLASS, etQuadratic> (aGrads, aPos, anInflVtx); break;
	case etPolynom4  : GradientAddAtInfluencedVerticesConfig<BLOBCLASS, etPolynom4 > (aGrads, aPos, anInflVtx); break;
	case etPolynom6  : GradientAddAtInfluencedVerticesConfig<BLOBCLASS, etPolynom6 > (aGrads, aPos, anInflVtx); break;
	default: 
		assert(0); 
		break;
	}
}

                           /////////////////////////////

template <class BLOBCLASS>
void BlobTransformable::GradientAddAtSubArrayFillInfluencedClass( Vect3fArray& aGrads, const Vect3DArray& aPos,
	                                                              VertexIndexArray& anInflVtxToFind,
	                                                              int32 nStart, int32 nCount) const
{
	switch( GetStyle() )
	{
	case etQuadratic : GradientAddAtSubArrayFillInfluencedConfig<BLOBCLASS, etQuadratic> (aGrads, aPos, anInflVtxToFind, nStart, nCount); break;
	case etPolynom4  : GradientAddAtSubArrayFillInfluencedConfig<BLOBCLASS, etPolynom4 > (aGrads, aPos, anInflVtxToFind, nStart, nCount); break;
	case etPolynom6  : GradientAddAtSubArrayFillInfluencedConfig<BLOBCLASS, etPolynom6 > (aGrads, aPos, anInflVtxToFind, nStart, nCount); break;
	default:
		assert(0);
		break;
	}
}

                           /////////////////////////////


template <class BLOBCLASS>
void BlobTransformable::FindInfluencedVerticesSubArrayClass( VertexIndexArray& anVtxInfluencedIdx, const Vect3DArray& aPos,
	                                                          int32 nStart, int32 nCount) const
{
	const BLOBCLASS& thisObj = static_cast<const BLOBCLASS&>(*this);
	anVtxInfluencedIdx.resize(0);

	int32 nEndNormals = nStart + nCount;
	for( int32 i = nStart; i < nEndNormals; ++i)
	{
		const Vector3D& pos = aPos[i];

		if( thisObj.BLOBCLASS::EarlyRejectPoint(pos) )
			continue;

		Vector3D vcOffs = pos;
		m_matTransfBaseShapeInv.Transform(vcOffs );
/*
		Vector3D vcOffs; 
		vcOffs.x = m_matTransfBaseShapeInv[MatrixTransf::M11] * pos.x +
					m_matTransfBaseShapeInv[MatrixTransf::M12] * pos.y +
					m_matTransfBaseShapeInv[MatrixTransf::M13] * pos.z +
					m_matTransfBaseShapeInv[MatrixTransf::M14];
		vcOffs.y = m_matTransfBaseShapeInv[MatrixTransf::M21] * pos.x +
					m_matTransfBaseShapeInv[MatrixTransf::M22] * pos.y +
					m_matTransfBaseShapeInv[MatrixTransf::M23] * pos.z +
					m_matTransfBaseShapeInv[MatrixTransf::M24];
		vcOffs.z = m_matTransfBaseShapeInv[MatrixTransf::M31] * pos.x +
					m_matTransfBaseShapeInv[MatrixTransf::M32] * pos.y +
					m_matTransfBaseShapeInv[MatrixTransf::M33] * pos.z +
					m_matTransfBaseShapeInv[MatrixTransf::M34];
*/
		if( thisObj.BLOBCLASS::EarlyRejectBaseShapePoint(vcOffs) )
			continue;

		float fSqrDist = thisObj.BLOBCLASS::GetBaseShapeSqrDist(vcOffs);
		
		if( fSqrDist >= 1.f )
			continue;

		anVtxInfluencedIdx.push_back( i );
	}
}

                           /////////////////////////////


template <class BLOBCLASS, int EVALTYPE>
inline void BlobTransformable::EnergyGetAtInfluencedVerticesConfig( VertexInfluenceArray& afVals, const Vect3DArray& aPos,
                                                                    const VertexIndexArray& anVertexIdx) const
{
	const BLOBCLASS& thisObj = static_cast<const BLOBCLASS&>(*this);

	int32 nVertexCount = anVertexIdx.size();
	afVals.resize( nVertexCount );
	
//	afVals.resize( 0 );
//	afVals.reserve( nVertexCount );


	for( int32 i = 0; i < nVertexCount; ++i)
	{
		float& fDest = afVals[i];
		int32 nIndex = anVertexIdx[i];
		const Vector3D& pos = aPos[nIndex];

//		Vector3D vcOffs = m_matTransfBaseShapeInv * pos; 
		Vector3D vcOffs = pos; m_matTransfBaseShapeInv.Transform(vcOffs); 
/*
		vcOffs.x = m_matTransfBaseShapeInv[MatrixTransf::M11] * pos.x +
				m_matTransfBaseShapeInv[MatrixTransf::M12] * pos.y +
				m_matTransfBaseShapeInv[MatrixTransf::M13] * pos.z +
				m_matTransfBaseShapeInv[MatrixTransf::M14];
		vcOffs.y = m_matTransfBaseShapeInv[MatrixTransf::M21] * pos.x +
				m_matTransfBaseShapeInv[MatrixTransf::M22] * pos.y +
				m_matTransfBaseShapeInv[MatrixTransf::M23] * pos.z +
				m_matTransfBaseShapeInv[MatrixTransf::M24];
		vcOffs.z = m_matTransfBaseShapeInv[MatrixTransf::M31] * pos.x +
				m_matTransfBaseShapeInv[MatrixTransf::M32] * pos.y +
				m_matTransfBaseShapeInv[MatrixTransf::M33] * pos.z +
				m_matTransfBaseShapeInv[MatrixTransf::M34];
*/
		float fSqrDist = thisObj.BLOBCLASS::GetBaseShapeSqrDist(vcOffs);
		fDest = EnergyNotZeroGetAtSqrDistConfig<EVALTYPE>(fSqrDist);
//		afVals.push_back( fDest );
	}
}

                           /////////////////////////////


template <class BLOBCLASS, int EVALTYPE>
inline bool BlobTransformable::GradientAddAtPosConfig( Vector3f& vcGrad, const Vector3D& pos) const
{
	const BLOBCLASS& thisObj = static_cast<const BLOBCLASS&>(*this);

	if( thisObj.BLOBCLASS::EarlyRejectPoint(pos) )
		return false;

//	Vector3D vcOffs = m_matTransfBaseShapeInv * pos; 
	Vector3D vcOffs = pos; m_matTransfBaseShapeInv.Transform(vcOffs); 
/*
	vcOffs.x = m_matTransfBaseShapeInv[MatrixTransf::M11] * pos.x +
	            m_matTransfBaseShapeInv[MatrixTransf::M12] * pos.y +
	            m_matTransfBaseShapeInv[MatrixTransf::M13] * pos.z +
	            m_matTransfBaseShapeInv[MatrixTransf::M14];
	vcOffs.y = m_matTransfBaseShapeInv[MatrixTransf::M21] * pos.x +
	            m_matTransfBaseShapeInv[MatrixTransf::M22] * pos.y +
	            m_matTransfBaseShapeInv[MatrixTransf::M23] * pos.z +
	            m_matTransfBaseShapeInv[MatrixTransf::M24];
	vcOffs.z = m_matTransfBaseShapeInv[MatrixTransf::M31] * pos.x +
	            m_matTransfBaseShapeInv[MatrixTransf::M32] * pos.y +
	            m_matTransfBaseShapeInv[MatrixTransf::M33] * pos.z +
	            m_matTransfBaseShapeInv[MatrixTransf::M34];
*/
	if( thisObj.BLOBCLASS::EarlyRejectBaseShapePoint(vcOffs) )
		return false;

	float fSqrDist = thisObj.BLOBCLASS::GetBaseShapeSqrDist(vcOffs);
	
	if( fSqrDist >= 1.f )
		return false;

	thisObj.BLOBCLASS::ConvertBaseShapePosToGradient(vcOffs);
	
	float fFactor = GradientGetVectorFactorNotZeroConfig< EVALTYPE >(fSqrDist);

	vcOffs.x *= fFactor;
	vcOffs.y *= fFactor;
	vcOffs.z *= fFactor;

	// Idem avec la multiplication de matrices (de plus pas de translation) : 
//    vcGrad += m_matTransfBaseShapeNormals * vcOffs;
	m_matTransfBaseShapeNormals.TransformDirection( vcOffs );
	vcGrad += Vector3f( float(vcOffs.x), float(vcOffs.y), float(vcOffs.z) );


/*
	vcGrad.x += m_matTransfBaseShapeNormals[MatrixTransf::M11] * vcOffs.x + 
		        m_matTransfBaseShapeNormals[MatrixTransf::M12] * vcOffs.y +
		        m_matTransfBaseShapeNormals[MatrixTransf::M13] * vcOffs.z;
	vcGrad.y += m_matTransfBaseShapeNormals[MatrixTransf::M21] * vcOffs.x + 
		        m_matTransfBaseShapeNormals[MatrixTransf::M22] * vcOffs.y +
		        m_matTransfBaseShapeNormals[MatrixTransf::M23] * vcOffs.z;
	vcGrad.z += m_matTransfBaseShapeNormals[MatrixTransf::M31] * vcOffs.x + 
		        m_matTransfBaseShapeNormals[MatrixTransf::M32] * vcOffs.y +
		        m_matTransfBaseShapeNormals[MatrixTransf::M33] * vcOffs.z;
*/
	return true;
}

                           /////////////////////////////


template <class BLOBCLASS, int EVALTYPE>
void BlobTransformable::GradientAddAtInfluencedVerticesConfig( Vect3fArray& aGrads, const Vect3DArray& aPos,
	                                                            const VertexIndexArray& anInflVtx) const
{
	const BLOBCLASS& thisObj = static_cast<const BLOBCLASS&>(*this);

	int32 nInfluencedVtxCount = anInflVtx.size();

	for( int32 i = 0; i < nInfluencedVtxCount; ++i)
	{
		int32 nIndex = anInflVtx[i];
		const Vector3D& pos = aPos[nIndex];
		Vector3f& vcGrad = aGrads[nIndex];

//		Vector3D vcOffs = m_matTransfBaseShapeInv * pos; 
		Vector3D vcOffs = pos; m_matTransfBaseShapeInv.Transform(vcOffs); 

/*
		vcOffs.x = m_matTransfBaseShapeInv[MatrixTransf::M11] * pos.x +
					m_matTransfBaseShapeInv[MatrixTransf::M12] * pos.y +
					m_matTransfBaseShapeInv[MatrixTransf::M13] * pos.z +
					m_matTransfBaseShapeInv[MatrixTransf::M14];
		vcOffs.y = m_matTransfBaseShapeInv[MatrixTransf::M21] * pos.x +
					m_matTransfBaseShapeInv[MatrixTransf::M22] * pos.y +
					m_matTransfBaseShapeInv[MatrixTransf::M23] * pos.z +
					m_matTransfBaseShapeInv[MatrixTransf::M24];
		vcOffs.z = m_matTransfBaseShapeInv[MatrixTransf::M31] * pos.x +
					m_matTransfBaseShapeInv[MatrixTransf::M32] * pos.y +
					m_matTransfBaseShapeInv[MatrixTransf::M33] * pos.z +
					m_matTransfBaseShapeInv[MatrixTransf::M34];
*/
		float fSqrDist = thisObj.BLOBCLASS::GetBaseShapeSqrDist(vcOffs);
		
		thisObj.BLOBCLASS::ConvertBaseShapePosToGradient(vcOffs);
		
		float fFactor = GradientGetVectorFactorNotZeroConfig< EVALTYPE >(fSqrDist);

		vcOffs.x *= fFactor;
		vcOffs.y *= fFactor;
		vcOffs.z *= fFactor;

		// Idem avec la multiplication de matrices (de plus pas de translation) : 
//        vcGrad += m_matTransfBaseShapeNormals * vcOffs;
		m_matTransfBaseShapeNormals.TransformDirection( vcOffs );
		vcGrad += Vector3f( float(vcOffs.x), float(vcOffs.y), float(vcOffs.z) );
		
/*
		vcGrad.x += m_matTransfBaseShapeNormals[MatrixTransf::M11] * vcOffs.x + 
					m_matTransfBaseShapeNormals[MatrixTransf::M12] * vcOffs.y +
					m_matTransfBaseShapeNormals[MatrixTransf::M13] * vcOffs.z;
		vcGrad.y += m_matTransfBaseShapeNormals[MatrixTransf::M21] * vcOffs.x + 
					m_matTransfBaseShapeNormals[MatrixTransf::M22] * vcOffs.y +
					m_matTransfBaseShapeNormals[MatrixTransf::M23] * vcOffs.z;
		vcGrad.z += m_matTransfBaseShapeNormals[MatrixTransf::M31] * vcOffs.x + 
					m_matTransfBaseShapeNormals[MatrixTransf::M32] * vcOffs.y +
					m_matTransfBaseShapeNormals[MatrixTransf::M33] * vcOffs.z;
*/
	}
}

                           /////////////////////////////


template <class BLOBCLASS, int EVALTYPE>
void BlobTransformable::GradientAddAtSubArrayFillInfluencedConfig( Vect3fArray& aGrads, const Vect3DArray& aPos,
	                                                               VertexIndexArray& anInflVtxToFind,
	                                                               int32 nStart, int32 nCount) const
{
	const BLOBCLASS& thisObj = static_cast<const BLOBCLASS&>(*this);

	int32 nEnd = nStart + nCount;

	for(int32 i = nStart; i < nEnd; ++i)
	{
		Vector3f& vcGrad = aGrads[i];
		const Vector3D& pos = aPos[i];

		if( thisObj.BLOBCLASS::EarlyRejectPoint(pos) )
			continue;

//		Vector3D vcOffs = m_matTransfBaseShapeInv * pos;
		Vector3D vcOffs = pos; m_matTransfBaseShapeInv.Transform(vcOffs); 
/*
		vcOffs.x = m_matTransfBaseShapeInv[MatrixTransf::M11] * pos.x +
		           m_matTransfBaseShapeInv[MatrixTransf::M12] * pos.y +
		           m_matTransfBaseShapeInv[MatrixTransf::M13] * pos.z +
		           m_matTransfBaseShapeInv[MatrixTransf::M14];
		vcOffs.y = m_matTransfBaseShapeInv[MatrixTransf::M21] * pos.x +
		           m_matTransfBaseShapeInv[MatrixTransf::M22] * pos.y +
		           m_matTransfBaseShapeInv[MatrixTransf::M23] * pos.z +
		           m_matTransfBaseShapeInv[MatrixTransf::M24];
		vcOffs.z = m_matTransfBaseShapeInv[MatrixTransf::M31] * pos.x +
		           m_matTransfBaseShapeInv[MatrixTransf::M32] * pos.y +
		           m_matTransfBaseShapeInv[MatrixTransf::M33] * pos.z +
		           m_matTransfBaseShapeInv[MatrixTransf::M34];
*/
		if( thisObj.BLOBCLASS::EarlyRejectBaseShapePoint(vcOffs) )
			continue;

		float fSqrDist = thisObj.BLOBCLASS::GetBaseShapeSqrDist(vcOffs);

		if( fSqrDist >= 1.f )
			continue;


		anInflVtxToFind.push_back( i );

		thisObj.BLOBCLASS::ConvertBaseShapePosToGradient(vcOffs);

		float fFactor = GradientGetVectorFactorNotZeroConfig< EVALTYPE >(fSqrDist);

		vcOffs.x *= fFactor;
		vcOffs.y *= fFactor;
		vcOffs.z *= fFactor;

		// Idem avec la multiplication de matrices (de plus pas de translation) : 
		m_matTransfBaseShapeNormals.TransformDirection( vcOffs ); 

		vcGrad += Vector3f( float(vcOffs.x), float(vcOffs.y), float(vcOffs.z) );

//		vcGrad += m_matTransfBaseShapeNormals * vcOffs;
/*
		vcGrad.x += m_matTransfBaseShapeNormals[MatrixTransf::M11] * vcOffs.x + 
		            m_matTransfBaseShapeNormals[MatrixTransf::M12] * vcOffs.y +
		            m_matTransfBaseShapeNormals[MatrixTransf::M13] * vcOffs.z;
		vcGrad.y += m_matTransfBaseShapeNormals[MatrixTransf::M21] * vcOffs.x + 
		            m_matTransfBaseShapeNormals[MatrixTransf::M22] * vcOffs.y +
		            m_matTransfBaseShapeNormals[MatrixTransf::M23] * vcOffs.z;
		vcGrad.z += m_matTransfBaseShapeNormals[MatrixTransf::M31] * vcOffs.x + 
		            m_matTransfBaseShapeNormals[MatrixTransf::M32] * vcOffs.y +
		            m_matTransfBaseShapeNormals[MatrixTransf::M33] * vcOffs.z;
*/
	}
}


                           /////////////////////////////












} // namespace


#endif	// #ifndef V3D_BLOBS_BLOB_TRANSFORMABLE_H_INCLUDED

